// scrape-agents.js
require('dotenv').config();
const fs        = require('fs');
const path      = require('path');
const os        = require('os');
const mysql     = require('mysql2/promise');
const readline  = require('readline');
const puppeteer = require('puppeteer-core');

const SOURCE_NAME = 'Sahibinden';
const dbConfig = {
  host:     'localhost',
  user:     'root',
  password: '',
  database: 'emlak_db',
  charset:  'utf8mb4'
};

// —— Config okumak için
const config = JSON.parse(fs.readFileSync(path.resolve(__dirname,'config.json'),'utf8'));
const { profileMaxAgeMinutes } = config;

// —— Profil klasörü yapılandırması
const PROFILE_ROOT = path.resolve(__dirname, 'profiles');
if (!fs.existsSync(PROFILE_ROOT)) fs.mkdirSync(PROFILE_ROOT, { recursive: true });

// Yaş tabanlı eski profilleri sil
function pruneOldProfiles(rootDir, maxAgeMinutes) {
  const now = Date.now();
  const maxAgeMs = maxAgeMinutes * 60 * 1000;
  for (const name of fs.readdirSync(rootDir)) {
    const full = path.join(rootDir, name);
    const stat = fs.statSync(full);
    if (stat.isDirectory() && now - stat.mtimeMs > maxAgeMs) {
      console.log(`🗑️ Siliniyor (yaş >${maxAgeMinutes}dk): ${name}`);
      fs.rmSync(full, { recursive: true, force: true });
    }
  }
}

// Her run için yeni profile dizini oluştur
function createProfile() {
  const sub = `profile-${Date.now()}`;
  const dir = path.join(PROFILE_ROOT, sub);
  fs.mkdirSync(dir, { recursive: true });
  return dir;
}

// İnsanlaştırma yardımcıları
const humanDelay = (min=1500, max=4000) =>
  new Promise(r => setTimeout(r, Math.random()*(max-min)+min));

const UAS = [
  'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 Chrome/115.0.0.0 Safari/537.36',
  'Mozilla/5.0 (Macintosh; Intel Mac OS X 13_4) AppleWebKit/537.36 Chrome/114.0.0.0 Safari/537.36',
  'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 Chrome/116.0.0.0 Safari/537.36'
];
function randomUserAgent() {
  return UAS[Math.floor(Math.random()*UAS.length)];
}
function randomViewport() {
  return {
    width:  Math.floor(Math.random()*(1440-1024)+1024),
    height: Math.floor(Math.random()*(900-700)+700)
  };
}
async function humanScroll(page) {
  await page.evaluate(()=>{ window.scrollTo(0, Math.random()*document.body.scrollHeight/3); });
  await humanDelay(500,1000);
  await page.evaluate(()=>{ window.scrollTo(0, document.body.scrollHeight); });
  await humanDelay(500,1000);
}
async function humanMoveMouse(page) {
  const vp = page.viewport();
  const x  = Math.random()*(vp.width-20);
  const y  = Math.random()*(vp.height-20);
  await page.mouse.move(x,y);
  await humanDelay(200,600);
}

// Konsoldan ENTER beklemek için
function waitForEnter(promptText='') {
  return new Promise(resolve => {
    const rl = readline.createInterface({ input: process.stdin, output: process.stdout });
    rl.question(promptText, () => { rl.close(); resolve(); });
  });
}

;(async () => {
  // 0) Eski profilleri temizle
  pruneOldProfiles(PROFILE_ROOT, profileMaxAgeMinutes);

  // 1) MySQL’e bağlan
  const conn = await mysql.createConnection(dbConfig);
  console.log('✅ MySQL’e bağlanıldı.');

  // 2) pendingLinks.json’dan linkleri al
  const linksPath = path.resolve(__dirname, 'pendingLinks.json');
  if (!fs.existsSync(linksPath)) {
    console.error('Error: pendingLinks.json bulunamadı.');
    process.exit(1);
  }
  const detailLinks = JSON.parse(fs.readFileSync(linksPath,'utf8'));

  // 3) Yeni bir profil diziniyle Puppeteer’a bağlan
  const profileDir = createProfile();
  console.log('🔑 Yeni profil:', profileDir);

  const browser = await puppeteer.connect({
    browserURL: 'http://127.0.0.1:9222',
    defaultViewport: null,
    slowMo: 50,
    protocolTimeout: 60000
  });
  let [page] = await browser.pages();
  if (!page) page = await browser.newPage();

  // Başlangıçta UA/Viewport ayarla
  await page.setUserAgent(randomUserAgent());
  await page.setViewport(randomViewport());

  // 4) Sahibinden ana sayfa, Cloudflare kontrolü
  console.log('▶ Sahibinden ana sayfa yükleniyor…');
  await page.goto('https://www.sahibinden.com', { waitUntil: 'domcontentloaded', timeout: 60000 });
  await humanDelay();
  if (page.url().includes('secure.sahibinden.com')) {
    console.log('⚠️ Cloudflare doğrulaması: Tarayıcıda tamamlayıp ENTER’a basın.');
    await waitForEnter('👉 ENTER ');
  }

  // 5) Detay linkleri döngüsü
  for (const detailUrl of detailLinks) {
    // insan hareketleri & UA rotasyonu
    await page.setUserAgent(randomUserAgent());
    await page.setViewport(randomViewport());

    console.log('\n▶ Detay:', detailUrl);
    try {
      await page.goto(detailUrl, { waitUntil: 'domcontentloaded', timeout: 45000 });
    } catch (e) {
      console.warn('  ⚠️ Detay sayfası yüklenemedi, atlanıyor…');
      continue;
    }
    await humanDelay();
    await humanScroll(page);
    await humanMoveMouse(page);

    // Cloudflare takılırsa 30s bekle
    if (page.url().includes('secure.sahibinden.com')) {
      console.log('  ⚠️ Cloudflare doğrulandı, 30s bekleniyor…');
      await page.waitForTimeout(30000);
    }

    // 6) Acente linkini almadan önce bekle
    await page.waitForSelector(
      'div.user-info-store-name > a, #classifiedDetail .classifiedContactSeller a',
      { timeout: 10000 }
    ).catch(() => {});
    let agentHref = await page.$eval(
      'div.user-info-store-name > a',
      a=>a.href
    ).catch(()=>null);
    if (!agentHref) {
      agentHref = await page.$eval(
        '#classifiedDetail .classifiedContactSeller a',
        a=>a.href
      ).catch(()=>null);
    }
    if (!agentHref) {
      console.log('  ⚠️ Acente linki bulunamadı.');
      continue;
    }
    console.log('  → Acente URL:', agentHref);

    // 7) Acenta sayfası: isim + konum
    try {
      await page.goto(agentHref, { waitUntil: 'domcontentloaded', timeout: 30000 });
    } catch {
      console.warn('  ⚠️ Acenta sayfası yüklenemedi, atlanıyor…');
      continue;
    }
    await humanDelay(800,1500);
    const storeDiv = await page.$('div.contact-info div.store-name');
    const agencyName = storeDiv
      ? await storeDiv.evaluate(el=>el.textContent.trim())
      : '';
    const location = storeDiv
      ? await storeDiv.evaluate(el=>el.getAttribute('data-main-location'))
      : '';
    console.log(`  → ${agencyName} — ${location}`);

    // 8) agents tablosuna kaydet (ON DUPLICATE)
    const [ins] = await conn.execute(
      `INSERT INTO agents (source, agency_name, agent_url, location)
       VALUES(?,?,?,?)
       ON DUPLICATE KEY UPDATE
         agency_name=VALUES(agency_name),
         location   =VALUES(location)`,
      [SOURCE_NAME, agencyName, agentHref, location]
    );
    let agentId = ins.insertId;
    if (!agentId) {
      const [[row]] = await conn.execute(
        `SELECT id FROM agents WHERE agent_url=? LIMIT 1`,
        [agentHref]
      );
      agentId = row.id;
    }

    // 9) Ekibimiz sayfası
    const { origin } = new URL(agentHref);
    const teamUrl = `${origin}/ekibimiz`;
    console.log('  ▶ Ekibimiz:', teamUrl);
    try {
      await page.goto(teamUrl, { waitUntil: 'domcontentloaded', timeout: 30000 });
    } catch {
      console.warn('  ⚠️ Ekibimiz sayfası yüklenemedi, atlanıyor…');
      continue;
    }
    await humanDelay();

    // 10) Ekip üyelerini çek & team_members’e kaydet
    const members = await page.$$eval(
      'div.edr-trigger.consultant',
      cards => cards.map(c=>({
        name:  c.querySelector('p.name')?.innerText.trim()  || '',
        tag:   c.querySelector('p.title')?.innerText.trim() || '',
        phone: c.querySelector('a.phone-trigger')?.getAttribute('data-phone') || ''
      }))
    ).catch(()=>[]);

    for (const m of members) {
      await conn.execute(
        `INSERT INTO team_members (agent_id, name, tag, phone)
         VALUES(?,?,?,?)
         ON DUPLICATE KEY UPDATE phone=VALUES(phone)`,
        [agentId, m.name, m.tag, m.phone]
      );
    }
    console.log(`  ✅ ${members.length} ekip üyesi kaydedildi.`);
  }

  // 11) Temizlik
  console.log('\n🎉 İşlem tamamlandı. Bağlantılar kapanıyor.');
  await browser.disconnect();
  await conn.end();
})();
